<?php

/**
 * Makes some changes to the <title> tag, by filtering the output of wp_title().
 *
 * If we have a site description and we're viewing the home page or a blog posts
 * page (when using a static front page), then we will add the site description.
 *
 * If we're viewing a search result, then we're going to recreate the title entirely.
 * We're going to add page numbers to all titles as well, to the middle of a search
 * result title and the end of all other titles.
 *
 * The site title also gets added to all titles.
 *
 * @since Expedia Theme 1.0
 *
 * @param string $title Title generated by wp_title()
 * @param string $separator The separator passed to wp_title(). Expedia Theme uses a
 * 	vertical bar, "|", as a separator in header.php.
 * @return string The new title, ready for the <title> tag.
 * @author WordPress Team
 */
function expediawp_filter_wp_title( $title, $separator ) {
	// Don't affect wp_title() calls in feeds.
	if ( is_feed() )
		return $title;

	// The $paged global variable contains the page number of a listing of posts.
	// The $page global variable contains the page number of a single post that is paged.
	// We'll display whichever one applies, if we're not looking at the first page.
	global $paged, $page;

	if ( is_search() ) {
		// If we're a search, let's start over:
		$title = sprintf( __( 'Search results for %s', 'expediawp' ), '"' . get_search_query() . '"' );
		// Add a page number if we're on page 2 or more:
		if ( $paged >= 2 )
			$title .= " $separator " . sprintf( __( 'Page %s', 'expediawp' ), $paged );
		// Add the site name to the end:
		$title .= " $separator " . get_bloginfo( 'name', 'display' );
		// We're done. Let's send the new title back to wp_title():
		return $title;
	}

	// Otherwise, let's start by adding the site name to the end:
	$title .= get_bloginfo( 'name', 'display' );

	// If we have a site description and we're on the home/front page, add the description:
	$site_description = get_bloginfo( 'description', 'display' );
	if ( $site_description && ( is_home() || is_front_page() ) )
		$title .= " $separator " . $site_description;

	// Add a page number if necessary:
	if ( $paged >= 2 || $page >= 2 )
		$title .= " $separator " . sprintf( __( 'Page %s', 'expediawp' ), max( $paged, $page ) );

	// Return the new title to wp_title():
	return $title;
}
add_filter( 'wp_title', 'expediawp_filter_wp_title', 10, 2 );

/**
 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
 *
 * @since Twenty Ten 1.0
 * @author WordPress Team
 */
function expediawp_page_menu_args( $args ) {
	$args['show_home'] = true;
	return $args;
}
add_filter( 'wp_page_menu_args', 'expediawp_page_menu_args' );

/**
 * Sets the post excerpt length to 80 characters.
 *
 * @since Expedia Theme 1.0
 * @return int
 * @author WordPress Team 
 */
function expediawp_excerpt_length( $length ) {
	return 80;
}
add_filter( 'excerpt_length', 'expediawp_excerpt_length' );

/**
 * Returns a "Read More" link for excerpts
 *
 * @since Expedia Theme 1.0
 * @return string "Continue Reading" link
 * @author Based WordPress Team Twenty Ten theme
 */
function expediawp_continue_reading_link() {
	return '<div class="read_more"><a href="'. get_permalink() . '">read more</a></div>';
}

/**
 * Replaces "[...]" (appended to automatically generated excerpts) with an ellipsis and expediawp_continue_reading_link().
 *
 * To override this in a child theme, remove the filter and add your own
 * function tied to the excerpt_more filter hook.
 *
 * @since Expedia Theme 1.0
 * @return string An ellipsis
 * @author WordPress Team 
 */
function expediawp_auto_excerpt_more( $more ) {
	return ' &hellip;' . expediawp_continue_reading_link();
}
add_filter( 'excerpt_more', 'expediawp_auto_excerpt_more' );

/**
 * Adds a pretty "Continue Reading" link to custom post excerpts.
 *
 * To override this link in a child theme, remove the filter and add your own
 * function tied to the get_the_excerpt filter hook.
 *
 * @since Expedia Theme 1.0
 * @return string Excerpt with a pretty "Continue Reading" link
 * @author WordPress Team  
 */
function expediawp_custom_excerpt_more( $output ) {
	if ( has_excerpt() && ! is_attachment() ) {
		$output .= expediawp_continue_reading_link();
	}
	return $output;
}
add_filter( 'get_the_excerpt', 'expediawp_custom_excerpt_more' );

// Trims excerpt to set number of characters looking for the last word before trimming.
function exp_custom_excerpt($excerpt, $excerpt_max_lenght = 60 ) {
	$excerpt = substr($excerpt,0,($excerpt_max_lenght-strlen(strrchr(substr($excerpt,0,$excerpt_max_lenght)," ")) )); 
	echo $excerpt .= '... ';
	
	return $excerpt;
}

/**
 * Content feature block. Content displays when a post is checked as "feature" in back end.
 *
 * @since Expedia Theme 1.0
 */
function exp_featured_post( $showposts = 1 ){
	global $wpdb,$cat,$post;
		$post_count=0;
		if(is_single()) {
			$post_cat = wp_get_post_categories($post->ID);
			$cat = $post_cat[0];
		}
		// Check for contextual featured post
		if($cat) {
			$query = sprintf("
			 SELECT DISTINCT %1\$s.*
			    FROM %1\$s
					JOIN %2\$s ON(%1\$s.ID = %2\$s.post_id)
					JOIN %3\$s ON(%1\$s.ID = %3\$s.object_id)
					JOIN %4\$s ON(%3\$s.term_taxonomy_id = %4\$s.term_taxonomy_id)
					JOIN %5\$s ON(%5\$s.term_id = %4\$s.term_id)
			    WHERE %1\$s.ID = %2\$s.post_id 
			    AND %2\$s.meta_key = 'exp_cat_feature'
			    AND %1\$s.post_status = 'publish' 
			    AND %1\$s.post_type = 'post'
				AND %5\$s.term_id = %6\$d
				ORDER BY %1\$s.post_date DESC
				LIMIT %7\$d
			 ",
			mysql_real_escape_string($wpdb->posts),
			mysql_real_escape_string($wpdb->postmeta),
			mysql_real_escape_string($wpdb->term_relationships),
			mysql_real_escape_string($wpdb->term_taxonomy),
			mysql_real_escape_string($wpdb->terms),
			mysql_real_escape_string($cat),
			mysql_real_escape_string($showposts)
			);
			$posts = $wpdb->get_results($query, OBJECT);
			$post_count = count($posts);
		}
		// If featured not in same context as viewed page, load latest featured
		if(!$post_count > 0 && (is_home() || is_front_page()) ) {
			$query = sprintf("
			    SELECT wposts.*,wpostmeta.*
			    FROM %s wposts, %s wpostmeta
			    WHERE wposts.ID = wpostmeta.post_id 
			    AND wpostmeta.meta_key = 'exp_main_feature'
			    AND wposts.post_status = 'publish' 
			    AND wposts.post_type = 'post'
			    ORDER BY wposts.post_date DESC
			    LIMIT %s
			 ",
			mysql_real_escape_string($wpdb->posts),
			mysql_real_escape_string($wpdb->postmeta),
			mysql_real_escape_string($showposts)
			);
		
			$posts = $wpdb->get_results($query, OBJECT);
			$post_count = count($posts);
		}
		$i=0;
		if ($post_count > 0 ) {
			echo '<div id="feature_box">';
			foreach ( $posts as $feature ) :
			setup_postdata($feature);
			?>
			<div class="feature_panel clearfix">
				<?php
				if ( $img = exp_get_meta_image('feature-thumbnail',$feature->ID) ) { ?>
						<a href="<?php echo get_permalink( $feature->ID ); ?>"><img src="<?php echo $img?>" alt="<?php echo $feature->post_title; ?>" /></a>
				<?php } ?>
					<h1><a href="<?php echo get_permalink( $feature->ID ); ?>"><?php echo $feature->post_title;?> </a></h1>
					<div class="entry-meta">
						<?php expediawp_posted_on($feature); ?>
					</div><!-- .entry-meta -->
					<p><?php
						exp_custom_excerpt(strip_tags(apply_filters('the_content', $feature->post_content)),'600');
					?>
					</p>
					<div class="read_more"><a href="<?php echo get_permalink( $feature->ID ); ?>">read more</a></div>
			</div>
			<?php
			endforeach;
			echo '<div class="feature_ft">&nbsp;</div>';
			echo '</div><!-- #feature_box -->';
		}
	wp_reset_query();
}

/**
 * Retrieves comments for blogroll
 *
 * @since Expedia Theme 1.0
 */
function exp_get_comments( $post_id = 0 , $shownumber = 2) { 
	global $wpdb, $id;
	
	if ( ! comments_open( $post_id ) ) {?>
		<div class="blogroll-comments"><?php _e( 'Comments are closed.', 'expediawp' ); ?></div>
	<?php 
		return;
	}
	
	
	$query = sprintf("
	    SELECT comment_content,comment_author,comment_ID,comment_post_ID
	    FROM %s
	    WHERE comment_post_ID = %s
	    AND comment_approved = '1'
	    ORDER BY comment_date DESC
	    LIMIT %s
	 ",
	mysql_real_escape_string($wpdb->comments),
	mysql_real_escape_string($post_id ),
	mysql_real_escape_string($shownumber)
	);
	$comments = $wpdb->get_results($query, OBJECT);
	$title_max_lenght = 20;
	$comment_max_lenght = 80;
	
	//print_r($comments);
	
	$post_link = get_permalink($post_id);
	
	if(empty($comments)) {
		echo '<div class="blogroll-comments">';
			echo 'There are no comments, <a href="'.$post_link.'#respond">click here</a> be the first one!';
		echo '</div>';
	
	} else {
		foreach ($comments as $comment) {
			$comment_excerpt = strip_tags($comment->comment_content);
			$comment_link = $post_link ."#comment-".$comment->comment_ID;
			// If has links, skip, don't want more links crowding the place
			if(strpos($comment_excerpt,'http://') > 0) 
				continue;
		
			// Sanitize text
			$comment_excerpt = preg_replace("/\[...\]/","",$comment_excerpt);	
			$comment_excerpt = preg_replace("/^<strong>.+?<\/strong>/","",$comment_excerpt,1);	
			$comment_excerpt = strip_tags(wptexturize($comment_excerpt));		
			$comment_excerpt = preg_replace("/[\n\t\r]/"," ",$comment_excerpt); // whitespace into 1 blank
			$comment_excerpt = preg_replace("/\s{2,}/"," ",$comment_excerpt); // whitespace into 1 blank
			
			if (strlen($comment_excerpt) >= $comment_max_lenght ) { $comment_excerpt = substr($comment_excerpt,0,($comment_max_lenght-strlen(strrchr(substr($comment_excerpt,0,$comment_max_lenght)," ")) )); $comment_excerpt .= '...'; }
		
			$comment_author = $comment->comment_author;
			if (!$comment_author) {
				$comment_author = "Anonymous";
			} else {
				if (strlen($comment_author) >= $title_max_lenght ) { $comment_author = substr($comment_author,0,($title_max_lenght-strlen(strrchr(substr($comment_author,0,$title_max_lenght)," ")) )); $comment_author .= '...'; }
			}
			
			echo '<div class="blogroll-comments">';
				echo '<strong>'.$comment_author.'</strong> <span>says:</span><br/>';
				echo $comment_excerpt.' <a href="'.$comment_link.'">more</a>';
			echo '</div>';
		}
		echo '<div class="read_more read_more-comments"><a href="'.$post_link.'#comments">read all</a></div>';

	} // endif
	 
?>
	
<?php }

function exp_footer_links() {
	$links = get_option('esp_theme_links');
	
	if(!$links) {
		$links = array(
			'1'=>array('url'=>'http://www.expedia.com/daily/deals/default.asp','kw'=>'Travel Deals'),
			'2'=>array('url'=>'http://www.expedia.com/Flights','kw'=>'Cheap Flights'),
			'3'=>array('url'=>'http://www.expedia.com/daily/cars/default.asp','kw'=>'Rental Cars'),
			'4'=>array('url'=>'http://www.expedia.com/Vacation-Packages','kw'=>'Trips'),
			'5'=>array('url'=>'http://www.expedia.com/','kw'=>'Airline Tickets'),
			'6'=>array('url'=>'http://www.expedia.com/Flights','kw'=>'Cheapest Flights'),
			'7'=>array('url'=>'http://cruise.expedia.com','kw'=>'Cruises'),
			'8'=>array('url'=>'http://www.expedia.com/','kw'=>'Travel'),
			'9'=>array('url'=>'http://www.expedia.com/Hotels','kw'=>'Hotel Reservations'),
			'10'=>array('url'=>'http://www.expedia.com/Hotels','kw'=>'Hotels')
		);
		$i = rand(1,10);
		add_option('esp_theme_links',$links[$i]);
	}
}

/**
 * Content output for Expedia Theme custom widget
 *
 * @since Expedia Theme 1.0
 */
function exp_widget_content( $category , $number ) {
	global $cat;
	if ($category == 'exp_widget_feature') {
		$wp_query = query_posts(array('showposts'=>$number,'meta_key'=>$category));
	} else {
		$category = strtolower($category);
		$id = get_cat_ID( $category);
		$wp_query = query_posts(array('showposts'=>$number,'cat'=>$id, 'post__not_in' => get_option( 'sticky_posts' ) ));
	}
	$count_posts = count($wp_query);
	$i=1;	
	if (have_posts()) {
		while (have_posts()) : the_post(); ?>
			<div class="listing">
					<?php
					$thumb = exp_get_meta_image('widget-thumbnail');
					if(!empty($thumb)) {echo '<a href="'.get_permalink().'"><img src="'.$thumb.'" class="attachment-widget-thumbnail" /></a>'; }
					?>
					<h2 class="entry-title"><a href="<?php the_permalink();?>"><?php the_title();?> </a></h2>
					<p><?php exp_custom_excerpt(get_the_excerpt(),'80'); ?><a href="<?php the_permalink();?>">read more</a></p>
			</div>
			<?php				
			if ($i == $count_posts && $category != 'exp_widget_feature') { echo '<p class="read_more"><a href="'.get_category_link($id).'">browse '.$category.'</a></p>'; }
			$i++;
		endwhile;
	} else { ?>
		<div class="listing">
			<?php _e('No posts found.', 'expediawp') ?>
		</div>
	<?php }
	wp_reset_query();
}

/**
 * Checks first for feature image, then scans post for embedded image within content and returns image path
 *
 * @param String $size - Either a string keyword (thumbnail, medium, large or full) or
 *        a 2-item array representing width and height in pixels, e.g. array(32,32). 
 * @since Expedia Theme 1.0
 */
function exp_get_meta_image($size = 'post-thumbnail',$post_id='') {
	/* Get image for post/page */
	global $post;
	
	$id = !empty($post_id) ? $post_id : $post->ID;
	// Kind of like using the_post_thumbnail(), but just want image path instead
	if ( $thumb_id = get_post_thumbnail_id($id) ) {
		$thumb = wp_get_attachment_image_src($thumb_id, $size);
		return $thumb[0];
	} else {
		return exp_get_img_by_scan(array('post_id'=>$id));
	}
}

/**
 * Scans post for embedded image within content and returns image path
 *
 * @since Expedia Theme 1.0
 */
function exp_get_img_by_scan( $args = array() ) {
	preg_match_all( '|<img.*?src=[\'"](.*?)[\'"].*?>|i', get_post_field( 'post_content', $args['post_id'] ), $matches );
	/* If there is a match for the image, return its URL. */
	if ( isset( $matches[1][0] ) ) {
		return $matches[1][0];
	}
}

/**
 * Custom search form
 *
 * @since Expedia Theme 1.0
 */
function exp_search_from() {
	$field_value = __('To search, type and hit enter', 'expediawp');
?>
	<form method="get" class="search_form" action="<?php echo home_url(); ?>/">
		<p>
			<input class="search_input" type="text" value="<?php echo $field_value; ?>" name="s" id="search" onfocus="if (this.value == '<?php echo $field_value; ?>') {this.value = '';}" onblur="if (this.value == '') {this.value = '<?php echo $field_value; ?>';}" />
			<input type="hidden" id="searchsubmit" value="Search" />
		</p>
	</form>
<?php }

/**
 * Prints HTML with meta information for the current post—date/time and author.
 *
 * @since Expedia Theme 1.0
 * @author Based WordPress Team Twenty Ten theme 
 */
function expediawp_posted_on($custom='') {
	global $post;
	
	if($custom)
		$post = $custom;
	
	$commentscount = get_comments_number($post->ID);
	if($commentscount == 1) $commenttext = ' comment';
	if($commentscount > 1 || $commentscount == 0) $commenttext = ' comments';
	
	if(comments_open( $post->ID ) || $commentscount > 0 ) {
		$comment = sprintf( '<span class="post_comments"><a href="%1$s#comments" alt="%2$s">%3$s</a></span>',
			get_permalink(),
			esc_attr__( 'View all comments', 'expediawp' ),
			sprintf( '[<em>%d</em>] %s', $commentscount ,$commenttext )
		);
	} else {
		$comment = sprintf( '<span class="post_comments">[<em>%d</em>] %s</span>', $commentscount ,$commenttext );	
	}
	
	printf( __( '<div class="posted">%3$s %2$s</div> %4$s', 'expediawp' ),
		'meta-sep meta-prep-author',
		sprintf( '<span class="entry-date"><a href="%1$s" title="%2$s">%3$s</a></span>',
			get_permalink(),
			esc_attr( get_the_time() ),
			get_the_date()
		),
		sprintf( '<span class="author vcard"><a class="url fn n" href="%1$s" title="%2$s">%3$s</a></span>',
			get_author_posts_url( get_the_author_meta( 'ID' ) ),
			sprintf( esc_attr__( 'View all posts by %s', 'expediawp' ), get_the_author() ),
			get_the_author()
		),
		$comment
	);
}

/**
 * Prints HTML with meta information for the current post (category, tags and permalink).
 *
 * @since Expedia Theme 1.0
 * @author Based WordPress Team Twenty Ten theme
 */
function expediawp_posted_in() {
	// Retrieves tag list of current post, separated by commas.
	$tag_list = get_the_tag_list( '', ', ' );
	if ( $tag_list ) {
		$posted_in = __( 'This entry was posted in %1$s and tagged %2$s. Bookmark the <a href="%3$s" title="Permalink to %4$s" rel="bookmark">permalink</a>.', 'expediawp' );
	} elseif ( is_object_in_taxonomy( get_post_type(), 'category' ) ) {
		$posted_in = __( 'This entry was posted in %1$s. Bookmark the <a href="%3$s" title="Permalink to %4$s" rel="bookmark">permalink</a>.', 'expediawp' );
	} else {
		$posted_in = __( 'Bookmark the <a href="%3$s" title="Permalink to %4$s" rel="bookmark">permalink</a>.', 'expediawp' );
	}
	// Prints the string, replacing the placeholders.
	printf(
		$posted_in,
		get_the_category_list( ', ' ),
		$tag_list,
		get_permalink(),
		the_title_attribute( 'echo=0' )
	);
}

/**
 * Template for comments and pingbacks.
 *
 * Used as a callback by wp_list_comments() for displaying the comments.
 *
 * @since Expedia Theme 1.0
 * @author Based WordPress Team Twenty Ten theme
 */
function expediawp_comment( $comment, $args, $depth ) {
	$GLOBALS['comment'] = $comment;
	switch ( $comment->comment_type ) :
		case '' :
	?>
	<li <?php comment_class(); ?> id="li-comment-<?php comment_ID(); ?>">
		<div id="comment-<?php comment_ID(); ?>">
		<div class="comment-author vcard">
			<?php echo get_avatar( $comment, 40 ); ?>
			<?php printf( __( '%s <span class="says">says:</span>', 'expediawp' ), sprintf( '<cite class="fn">%s</cite>', get_comment_author_link() ) ); ?>
		</div><!-- .comment-author .vcard -->
		<?php if ( $comment->comment_approved == '0' ) : ?>
			<em><?php _e( 'Your comment is awaiting moderation.', 'expediawp' ); ?></em>
			<br />
		<?php endif; ?>

		<div class="comment-meta commentmetadata"><a href="<?php echo esc_url( get_comment_link( $comment->comment_ID ) ); ?>">
			<?php
				/* translators: 1: date, 2: time */
				printf( __( '%1$s at %2$s', 'expediawp' ), get_comment_date(),  get_comment_time() ); ?></a><?php edit_comment_link( __( '(Edit)', 'expediawp' ), ' ' );
			?>
		</div><!-- .comment-meta .commentmetadata -->

		<div class="comment-body"><?php comment_text(); ?></div>

		<div class="reply">
			<?php comment_reply_link( array_merge( $args, array( 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
		</div><!-- .reply -->
	</div><!-- #comment-##  -->

	<?php
			break;
		case 'pingback'  :
		case 'trackback' :
	?>
	<li class="post pingback">
		<p><?php _e( 'Pingback:', 'expediawp' ); ?> <?php comment_author_link(); ?><?php edit_comment_link( __('(Edit)', 'expediawp'), ' ' ); ?></p>
	<?php
			break;
	endswitch;
}

function get_exp_social_links() {
	$follows = get_option('exp_social_feeds');
	$face = ($follows['facebook_like_url'] !='') ? $follows['facebook_like_url'] : site_url();
	$rss = ($follows['rss'] !='') ? $follows['rss'] : get_bloginfo('rss2_url');
	?>
	<div class="tweet">
		<a href="http://twitter.com/share" class="twitter-share-button" data-url="<?php echo site_url(); ?>" data-count="horizontal" data-via="" data-related="">Tweet</a>
	</div>
	<div class="googleplus">
		<g:plusone href="<?php echo site_url(); ?>"></g:plusone>
	</div>
	<div class="facebook">
		<fb:like href="<?php echo $face; ?>" layout="button_count" show_faces="false" width="110"></fb:like>
	</div>
	<div class="follow">
		<span>Follow:</span>
		<?php
			if($follows['facebook'])
				echo '<a href="'.$follows['facebook'].'" title="follow '.get_bloginfo('name').' on facebook"><img src="'.get_stylesheet_directory_uri().'/images/icons/facebook.png" alt="follow '.get_bloginfo('name').' on facebook" /></a>';
			
			if($follows['twitter'])
				echo '<a href="http://twitter.com/'.$follows['twitter'].'" title="follow '.get_bloginfo('name').' on twitter"><img src="'.get_stylesheet_directory_uri().'/images/icons/twitter.png" alt="follow '.get_bloginfo('name').' on twitter" /></a>';
				
			echo '<a href="'.$rss.'" title="follow '.get_bloginfo('name').' via RSS"><img src="'.get_stylesheet_directory_uri().'/images/icons/rss.png" alt="follow '.get_bloginfo('name').' via RSS" /></a>';
			
		?>
	</div>
<?php }


function twentyten_remove_gallery_css( $css ) {
	return preg_replace( "#<style type='text/css'>(.*?)</style>#s", '', $css );
}
add_filter( 'gallery_style', 'twentyten_remove_gallery_css' );

// Font list for admin page
function exp_font_list() {
	$fonts = array(
			'arial' => array(
				'name' => 'Arial',
				'family' => 'Arial, "Helvetica Neue", Helvetica, sans-serif',
				'web_safe' => true
			),
			'arial_black' => array(
				'name' => 'Arial Black',
				'family' => '"Arial Black", "Arial Bold", Arial, sans-serif',
				'web_safe' => true
			),
			'arial_narrow' => array(
				'name' => 'Arial Narrow',
				'family' => '"Arial Narrow", Arial, "Helvetica Neue", Helvetica, sans-serif',
				'web_safe' => true
			),
			'courier_new' => array(
				'name' => 'Courier New',
				'family' => '"Courier New", Courier, Verdana, sans-serif',
				'web_safe' => true,
				'monospace' => true
			),
			'georgia' => array(
				'name' => 'Georgia',
				'family' => 'Georgia, "Times New Roman", Times, serif',
				'web_safe' => true
			),
			'times_new_roman' => array(
				'name' => 'Times New Roman',
				'family' => '"Times New Roman", Times, Georgia, serif',
				'web_safe' => true
			),
			'trebuchet_ms' => array(
				'name' => 'Trebuchet MS',
				'family' => '"Trebuchet MS", "Lucida Grande", "Lucida Sans Unicode", "Lucida Sans", Arial, sans-serif',
				'web_safe' => true
			),
			'verdana' => array(
				'name' => 'Verdana',
				'family' => 'Verdana, sans-serif',
				'web_safe' => true
			),
			'andale' => array(
				'name' => 'Andale Mono',
				'family' => '"Andale Mono", Consolas, Monaco, Courier, "Courier New", Verdana, sans-serif',
				'web_safe' => false,
				'monospace' => true
			),
			'baskerville' => array(
				'name' => 'Baskerville',
				'family' => 'Baskerville, "Times New Roman", Times, serif',
				'web_safe' => false
			),
			'bookman_old_style' => array(
				'name' => 'Bookman Old Style',
				'family' => '"Bookman Old Style", Georgia, "Times New Roman", Times, serif',
				'web_safe' => false
			),
			'calibri' => array(
				'name' => 'Calibri',
				'family' => 'Calibri, "Helvetica Neue", Helvetica, Arial, Verdana, sans-serif',
				'web_safe' => false
			),
			'cambria' => array(
				'name' => 'Cambria',
				'family' => 'Cambria, Georgia, "Times New Roman", Times, serif',
				'web_safe' => false
			),
			'candara' => array(
				'name' => 'Candara',
				'family' => 'Candara, Verdana, sans-serif',
				'web_safe' => false
			),
			'century_gothic' => array(
				'name' => 'Century Gothic',
				'family' => '"Century Gothic", "Apple Gothic", Verdana, sans-serif',
				'web_safe' => false
			),
			'century_schoolbook' => array(
				'name' => 'Century Schoolbook',
				'family' => '"Century Schoolbook", Georgia, "Times New Roman", Times, serif',
				'web_safe' => false
			),
			'consolas' => array(
				'name' => 'Consolas',
				'family' => 'Consolas, "Andale Mono", Monaco, Courier, "Courier New", Verdana, sans-serif',
				'web_safe' => false,
				'monospace' => true
			),
			'constantia' => array(
				'name' => 'Constantia',
				'family' => 'Constantia, Georgia, "Times New Roman", Times, serif',
				'web_safe' => false
			),
			'corbel' => array(
				'name' => 'Corbel',
				'family' => 'Corbel, "Lucida Grande", "Lucida Sans Unicode", Arial, sans-serif',
				'web_safe' => false
			),
			'franklin_gothic' => array(
				'name' => 'Franklin Gothic Medium',
				'family' => '"Franklin Gothic Medium", Arial, sans-serif',
				'web_safe' => false
			),
			'garamond' => array(
				'name' => 'Garamond',
				'family' => 'Garamond, "Hoefler Text", "Times New Roman", Times, serif',
				'web_safe' => false
			),
			'gill_sans' => array(
				'name' => 'Gill Sans',
				'family' => '"Gill Sans MT", "Gill Sans", Calibri, "Trebuchet MS", sans-serif',
				'web_safe' => false
			),
			'helvetica' => array(
				'name' => 'Helvetica',
				'family' => '"Helvetica Neue", Helvetica, Arial, sans-serif',
				'web_safe' => false
			),
			'hoefler' => array(
				'name' => 'Hoefler Text',
				'family' => '"Hoefler Text", Garamond, "Times New Roman", Times, sans-serif',
				'web_safe' => false
			),
			'lucida_bright' => array(
				'name' => 'Lucida Bright',
				'family' => '"Lucida Bright", Cambria, Georgia, "Times New Roman", Times, serif',
				'web_safe' => false
			),
			'lucida_grande' => array(
				'name' => 'Lucida Grande',
				'family' => '"Lucida Grande", "Lucida Sans", "Lucida Sans Unicode", sans-serif',
				'web_safe' => false
			),
			'palatino' => array(
				'name' => 'Palatino',
				'family' => '"Palatino Linotype", Palatino, Georgia, "Times New Roman", Times, serif',
				'web_safe' => false
			),
			'rockwell' => array(
				'name' => 'Rockwell',
				'family' => 'Rockwell, "Arial Black", "Arial Bold", Arial, sans-serif',
				'web_safe' => false
			),
			'tahoma' => array(
				'name' => 'Tahoma',
				'family' => 'Tahoma, Geneva, Verdana, sans-serif',
				'web_safe' => false
			)
		);
	return $fonts;
}

?>